import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';

import { updateInputState, updateColorsObj, updateWebsiteAddons } from 'stores/webDevelopment';

import withFlowManaging from 'helpers/withFlowManaging';
import {
  WEB_COLORS,
  WEB_COLORS_OPTIONS,
  WEB_ADDONS,
  WEB_STORE,
  WEB_PCR,
  WEB_PLAN,
  WEB_WEBSITES
} from 'helpers/constants';
import { isAnySelected } from 'helpers';

import StepWrapper from 'components/StepWrapper';
import FlexContainer from 'components/FlexContainer';
import Typography from 'components/Typography';
import FormGroup from 'components/FormGroup';
import ColorPicker from 'components/ColorPicker';
import Input from 'components/Input';
import Checkbox from 'components/Checkbox';
import ShopIcon from 'components/Icons/ShopIcon';
import BoardIcon from 'components/Icons/BoardIcon';
import ClockIcon from 'components/Icons/ClockIcon';

import commonStyles from 'containers/Common.module.scss';

const checkboxItems = [
  {
    id: WEB_STORE,
    label: 'Online Store',
    icon: <ShopIcon />
  },
  {
    id: WEB_PCR,
    label: 'PCR Job Board',
    icon: <BoardIcon />
  },
  {
    id: WEB_PLAN,
    label: 'Update Plan',
    icon: <ClockIcon />
  }
];

function DevelopmentDetails({
  colorsObj,
  websites,
  addonsObj,
  updateColors,
  updateWebsites,
  updateWebsiteAddons
}) {
  return (
    <StepWrapper
      isShort
      progressLabel={`Web Development: 2 / 3`}
      disableButton={isAnySelected(addonsObj)}
    >
      <Typography size="Huge" weight={700} marginBottom={24}>
        Help us create something you&apos;ll love
      </Typography>

      <FormGroup>
        <Typography marginBottom={8}>
          Choose your mockup&apos;s colors
        </Typography>
        <Typography isGray size="Small" weight={400} marginBottom={16}>
          If you don&apos;t know, simply leave it blank
        </Typography>
        <FlexContainer flexAlign="Between" className={commonStyles.OneToFour}>
          {WEB_COLORS_OPTIONS.map((id) => (
            <ColorPicker id={id} onClick={updateColors} key={id} currentColor={colorsObj&&colorsObj[id]}/>
          ))}
        </FlexContainer>
      </FormGroup>

      <FormGroup>
        <Typography marginBottom={8}>
          What websites have you seen that you like?
        </Typography>
        <Typography isGray size="Small" weight={400} marginBottom={16}>
          If you don&apos;t know, check out our portfolio!
        </Typography>
        <Input
          placeholder="e.g. carbonweb.co"
          id={WEB_WEBSITES}
          value={websites}
          onChange={(name, value) => {
            updateWebsites({name, value})
          }}
        />
      </FormGroup>

      <FormGroup>
        <Typography marginBottom={8}>
          Choose the add-ons that you would like
        </Typography>
        <Typography isGray size="Small" weight={400} marginBottom={16}>
          Add-ons may affect your website&apos;s tier
        </Typography>
        <FlexContainer flexAlign="Between" className={commonStyles.OneToThree}>
          {
            checkboxItems.map(({id, label, icon}) => (
              <Checkbox
                key={id}
                id={id}
                label={label}
                icon={icon}
                isSelected={addonsObj[id]}
                onClick={() => updateWebsiteAddons({name:id, value: !addonsObj[id]})}
              />
            ))
          }
        </FlexContainer>
      </FormGroup>
    </StepWrapper>
  );
}

const mapStateToProps = state => ({
  colorsObj: state.webDevelopment[WEB_COLORS],
  websites: state.webDevelopment[WEB_WEBSITES],
  addonsObj: state.webDevelopment[WEB_ADDONS]
});

const mapDispatchToProps = dispatch => ({
  updateColors: payload => dispatch(updateColorsObj(payload)),
  updateWebsites: payload => dispatch(updateInputState(payload)),
  updateWebsiteAddons: payload => dispatch(updateWebsiteAddons(payload))
});

DevelopmentDetails.propTypes = {
  colorsObj: PropTypes.object,
  websites: PropTypes.string,
  addonsObj: PropTypes.object,
  updateColors: PropTypes.func,
  updateWebsites: PropTypes.func,
  updateWebsiteAddons: PropTypes.func
};

export default connect(mapStateToProps, mapDispatchToProps)(withFlowManaging(DevelopmentDetails));

