import React, { useRef, useCallback, useEffect } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';

import withFlowManaging from 'helpers/withFlowManaging';
import {
  PD_FORM,
  PD_FINDOUT,
  PD_FINDOUT_OTHER_INPUT,
  PD_TIME_ZONE,
  PD_FINDOUT_OPTIONS
} from 'helpers/personalDetailsConstants';
import { isAnyEmptyInput } from 'helpers';

import { updatePersonalDetailsInput } from 'stores/personalDetails';
import { sendData, setHasError } from 'stores/api';

import StepWrapper from 'components/StepWrapper';
import Typography from 'components/Typography';
import FlexContainer from 'components/FlexContainer';
import FormGroup from 'components/FormGroup';
import RadioButton from 'components/RadioButton';
import Input from 'components/Input';
import Timezone from 'components/Input/Timezone';

import commonStyles from 'containers/Common.module.scss';

import { inputItems } from './inputItems';

function PersonalDetails({
  allInputs,
  timezone,
  findout,
  findoutOtherInput,
  updatePersonalDetailsInput,
  sendData,
  hasError,
  setHasError
}) {
  const formEl = useRef(null);
  const onInputUpdate = useCallback(payload => {
    updatePersonalDetailsInput(payload);
    setHasError(false);
  }, [updatePersonalDetailsInput, setHasError])
  const onSubmit = () => {
    if(formEl.current.reportValidity()) {
      setHasError(false); sendData();
    }
  };
  const clearErrors = useCallback(() => {setHasError(false)}, [setHasError]);

  useEffect(() => {
    window.addEventListener('beforeunload', clearErrors);

    return () => {
      clearErrors();
      window.removeEventListener('beforeunload', clearErrors);
    }
  }, [clearErrors]);

  return (
      <StepWrapper
        isShort
        buttonLabel="Send"
        progressLabel={"Main Contact Details"}
        disableButton={isAnyEmptyInput(allInputs)}
        nextButtonAction={onSubmit}
      >
        <Typography size="Huge" weight={700} marginBottom={16}>
          Main contact details
        </Typography>
        <form ref={formEl}>
          {inputItems.map(({label, placeholder, id, type}) => (
            <FormGroup key={label}>
              <Input
                type={type}
                label={label}
                placeholder={placeholder}
                id={id}
                value={allInputs[id]}
                onChange={(id, value) => {
                  onInputUpdate({id, value})
                }}
                required
              />
            </FormGroup>
          ))}

          <FormGroup>
            <Typography marginBottom={12}>
              How did you hear about us?
            </Typography>
            <FlexContainer flexAlign="Between" className={commonStyles.OneToTwo}>
              {PD_FINDOUT_OPTIONS.map((value) => (
                <RadioButton
                  key={value}
                  label={value}
                  name={PD_FINDOUT}
                  value={value}
                  isSelected={findout === value}
                  onClick={() => (onInputUpdate({id: PD_FINDOUT, value}))}
                />
              ))}
            </FlexContainer>
            {findout === PD_FINDOUT_OPTIONS[5]
            && <Input
              placeholder="Other"
              id={PD_FINDOUT_OTHER_INPUT}
              value={findoutOtherInput}
              onChange={(id, value) => {
                onInputUpdate({id, value})
              }}
              required
            />}
          </FormGroup>

          <FormGroup>
            <Typography marginBottom={12}>
              What is your time zone?
            </Typography>

            <Timezone
              value={timezone}
              handleOnChange={(value) => {
              onInputUpdate({id: PD_TIME_ZONE, value})
            }}/>
          </FormGroup>
        </form>

        {hasError && (
          <Typography marginBottom={12} className={commonStyles.ColorError}>
            Something went wrong. Please check your data and retry.
          </Typography>
        )}
      </StepWrapper>
  );
}

const mapStateToProps = state => ({
  allInputs: state.personalDetails[PD_FORM],
  findout: state.personalDetails[PD_FORM][PD_FINDOUT],
  findoutOtherInput: state.personalDetails[PD_FORM][PD_FINDOUT_OTHER_INPUT],
  timezone: state.personalDetails[PD_FORM][PD_TIME_ZONE],
  hasError: state.api.hasError
});

const mapDispatchToProps = dispatch => ({
  updatePersonalDetailsInput: payload => dispatch(updatePersonalDetailsInput(payload)),
  sendData: () => dispatch(sendData()),
  setHasError: (hasError) => dispatch(setHasError(hasError))
});

PersonalDetails.propTypes = {
  areDataSent: PropTypes.bool,
  allInputs: PropTypes.object,
  findout: PropTypes.string,
  findoutOtherInput: PropTypes.string,
  timezone: PropTypes.string,
  updatePersonalDetailsInput: PropTypes.func,
  sendData: PropTypes.func,
  hasError: PropTypes.bool,
  setHasError: PropTypes.func
};

export default connect(mapStateToProps, mapDispatchToProps)(withFlowManaging(PersonalDetails));

